/*-
 * Copyright (c) 2003-2007 Tim Kientzle
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR(S) ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR(S) BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "archive_platform.h"
__FBSDID("$FreeBSD: src/lib/libarchive/archive_util.c,v 1.17 2008/03/14 22:31:57 kientzle Exp $");

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "archive.h"
#include "archive_private.h"
#include "archive_string.h"

#if ARCHIVE_VERSION_NUMBER < 3000000
/* These disappear in libarchive 3.0 */
/* Deprecated. */
int
archive_api_feature(void)
{
	return (ARCHIVE_API_FEATURE);
}

/* Deprecated. */
int
archive_api_version(void)
{
	return (ARCHIVE_API_VERSION);
}

/* Deprecated synonym for archive_version_number() */
int
archive_version_stamp(void)
{
	return (archive_version_number());
}

/* Deprecated synonym for archive_version_string() */
const char *
archive_version(void)
{
	return (archive_version_string());
}
#endif

int
archive_version_number(void)
{
	return (ARCHIVE_VERSION_NUMBER);
}

/*
 * Format a version string of the form "libarchive x.y.z", where x, y,
 * z are the correct parts of the version ID from
 * archive_version_number().
 *
 * I used to do all of this at build time in shell scripts but that
 * proved to be a portability headache.
 */

const char *
archive_version_string(void)
{
	static char buff[128];
	struct archive_string as;
	int n;

	if (buff[0] == '\0') {
		n = archive_version_number();
		memset(&as, 0, sizeof(as));
		archive_string_sprintf(&as, "libarchive %d.%d.%d",
		    n / 1000000, (n / 1000) % 1000, n % 1000);
		strncpy(buff, as.s, sizeof(buff));
		buff[sizeof(buff) - 1] = '\0';
		archive_string_free(&as);
	}
	return (buff);
}

int
archive_errno(struct archive *a)
{
	return (a->archive_error_number);
}

const char *
archive_error_string(struct archive *a)
{

	if (a->error != NULL  &&  *a->error != '\0')
		return (a->error);
	else
		return ("(Empty error message)");
}


int
archive_format(struct archive *a)
{
	return (a->archive_format);
}

const char *
archive_format_name(struct archive *a)
{
	return (a->archive_format_name);
}


int
archive_compression(struct archive *a)
{
	return (a->compression_code);
}

const char *
archive_compression_name(struct archive *a)
{
	return (a->compression_name);
}


/*
 * Return a count of the number of compressed bytes processed.
 */
int64_t
archive_position_compressed(struct archive *a)
{
	return (a->raw_position);
}

/*
 * Return a count of the number of uncompressed bytes processed.
 */
int64_t
archive_position_uncompressed(struct archive *a)
{
	return (a->file_position);
}

void
archive_clear_error(struct archive *a)
{
	archive_string_empty(&a->error_string);
	a->error = NULL;
}

void
archive_set_error(struct archive *a, int error_number, const char *fmt, ...)
{
	va_list ap;
#ifdef HAVE_STRERROR_R
	char errbuff[512];
#endif
	char *errp;

	a->archive_error_number = error_number;
	if (fmt == NULL) {
		a->error = NULL;
		return;
	}

	va_start(ap, fmt);
	archive_string_vsprintf(&(a->error_string), fmt, ap);
	if (error_number > 0) {
		archive_strcat(&(a->error_string), ": ");
#ifdef HAVE_STRERROR_R
#ifdef STRERROR_R_CHAR_P
		errp = strerror_r(error_number, errbuff, sizeof(errbuff));
#else
		strerror_r(error_number, errbuff, sizeof(errbuff));
		errp = errbuff;
#endif
#else
		/* Note: this is not threadsafe! */
		errp = strerror(error_number);
#endif
		archive_strcat(&(a->error_string), errp);
	}
	a->error = a->error_string.s;
	va_end(ap);
}

void
archive_copy_error(struct archive *dest, struct archive *src)
{
	dest->archive_error_number = src->archive_error_number;

	archive_string_copy(&dest->error_string, &src->error_string);
	dest->error = dest->error_string.s;
}

void
__archive_errx(int retvalue, const char *msg)
{
	static const char *msg1 = "Fatal Internal Error in libarchive: ";
	write(2, msg1, strlen(msg1));
	write(2, msg, strlen(msg));
	write(2, "\n", 1);
	exit(retvalue);
}
